// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef OPARRAY_H
#define OPARRAY_H

#include "common.h"
#include "array.h"

template<class pOpfunc> class OpcodeBase {
public:
	virtual pOpfunc getOpFunc(DWORD opcode) const = 0;
	virtual const char *getOpAsm(DWORD opcode) const = 0;
	virtual void dump() const = 0;
	virtual ~OpcodeBase() {}
	virtual bool isArray() const = 0;
protected:
	static int dump_level;
};

template<class pOpfunc> class Opcode : public OpcodeBase<pOpfunc> {
public:
	Opcode(pOpfunc f, const char *a) : opfunc(f), opasm(a) {}

	pOpfunc getOpFunc(DWORD /*opcode*/) const { return opfunc; }
	const char *getOpAsm(DWORD /*opcode*/) const { return opasm; }
	void dump() const;
private:
	pOpfunc opfunc;
	const char *opasm;

	bool isArray() const { return false; }
};

template<class pOpfunc> class OpArray : public OpcodeBase<pOpfunc> {
public:
	OpArray(DWORD s, DWORD e);
	~OpArray();
	void addOpcode(pOpfunc f, const char *a, DWORD pri, DWORD sm, DWORD sec);

	//Consider optimizing by consolidating into structure
	pOpfunc getOpFunc(DWORD opcode) const;
	const char *getOpAsm(DWORD opcode) const;
	void dump() const;
private:
	DWORD start, end, mask;
	Array<OpcodeBase<pOpfunc> *> arr;

	bool isArray() const { return true; }
	DWORD getNum(DWORD opcode) const { return (opcode & mask) >> (31-end); }
};


template<class pOpfunc>
OpArray<pOpfunc>::OpArray(DWORD s, DWORD e) : start(s), end(e), arr(2 << (e - s)) {
	mask = makemaskw(start, end);
	//DEGUB("Array created. mask=%08X  size=%i\n", mask, arr.size());
	for(size_t i=0; i<arr.size(); i++)
		arr[i] = NULL;
}

template<class pOpfunc>
OpArray<pOpfunc>::~OpArray() {
	//DEGUB("Deleting array. mask=%08X  size=%i\n", mask, arr.size());
	for(size_t i=0; i<arr.size(); i++)
		if(arr[i])
			delete arr[i];
	//DEGUB("Array deleted. mask=%08X  size=%i\n", mask, arr.size());
}

//See notes.txt for opcode explanation
#define OPCODE(pri, sec) ((pri << 26) | (sec << 1))

template<class pOpfunc>
void OpArray<pOpfunc>::addOpcode(pOpfunc f, const char *a, DWORD pri, DWORD sm, DWORD sec) {
	DWORD opcode = OPCODE(pri, sec);
	DWORD num = getNum(opcode);

	//DEGUB("Adding %s (%i, %i, %i)  num=%i\n", a, pri, sm, sec, num);
	if(getOpAsm(opcode)) {
		DEGUB("%s found when trying to add %s!\n", getOpAsm(opcode), a);
	}
	MYASSERT(!getOpFunc(opcode));

	if(arr[num] == NULL) {
		if(start == sm) {
			arr[num] = new Opcode<pOpfunc>(f, a);
		} else {
			OpArray *temp;
			if(start == 0) {
				arr[num] = temp = new OpArray(sm, 30);
				temp->addOpcode(f, a, pri, sm, sec);
			} else if(start > sm) {
				arr[num] = temp = new OpArray(sm, start-1);
				temp->addOpcode(f, a, pri, sm, sec);
			} else if(start < sm) {
				DWORD i, j, divider = 1 << (sm - start), multiplier = 0x80000000 >> sm;
				Array<OpcodeBase<pOpfunc> *> temparr(0);
				temparr.steal(arr);

				arr.resize(2 << (end - sm));
				for(i=0; i<arr.size(); i++)
					arr[i] = NULL;

				for(i=0; i<arr.size(); i++) {
					temp = NULL;
					for(j=0; j<divider; j++) {
						if(!temp && temparr[j * multiplier + i])
							arr[i] = temp = new OpArray(start, sm-1);
						if(temp) {
							temp->arr[j] = temparr[j * multiplier + i];
							temparr[j * multiplier + i] = NULL;
						}
					}
				}

				//degub
				for(i=0; i<temparr.size(); i++) {
					if(temparr[i] != NULL) {
						DEGUB("Leak? %i of %i isarray:%i\n", i, temparr.size(), temparr[i]->isArray());
						delete temparr[i];
					}
				}

				start = sm;
				mask = makemaskw(start, end);
				addOpcode(f, a, pri, sm, sec);
			}	
		}
	} else {
		if(arr[num]->isArray()) {
			MAKEP(OpArray, arr[num])->addOpcode(f, a, pri, sm, sec);
		} else {
			BFE("Opcode array init error");
		}
	}
}

template<class pOpfunc>
pOpfunc OpArray<pOpfunc>::getOpFunc(DWORD opcode) const {
	DWORD num = getNum(opcode);
	if(arr[num])
		return arr[num]->getOpFunc(opcode);
	else
		return NULL;
}
template<class pOpfunc>
const char *OpArray<pOpfunc>::getOpAsm(DWORD opcode) const {
	DWORD num = getNum(opcode);
	if(arr[num])
		return arr[num]->getOpAsm(opcode);
	else
		return NULL;
}

template<class pOpfunc>
int OpcodeBase<pOpfunc>::dump_level = 0;

template<class pOpfunc>
void Opcode<pOpfunc>::dump() const {
	DEGUB("%s\n", opasm);
}
template<class pOpfunc>
void OpArray<pOpfunc>::dump() const {
	DEGUB("Array | level %i | start %i | end %i | mask 0x%08X | size %i\n",
		dump_level, start, end, mask, arr.size());
	dump_level++;
	for(size_t i=0; i<arr.size(); i++) {
		if(arr[i]) {
			dump_spaces(dump_level);
			DEGUB("%i: ", i);
			arr[i]->dump();
		}
	}
	dump_level--;
}

#endif	//OPARRAY_H
